#include "local_task_scheduler.h"

local_task::local_task(int id, std::shared_ptr<meta_task> meta)
    : id(id), meta(meta)
{  }

bool local_task::work()
{
    kernel::compute(meta, id);

    int temp = ++(meta->fulfilled);

    if (temp == meta->nlocal)
    {
        return true;
    }
    else if (temp > meta->nlocal)
    {
        std::cout << "error: more than nlocal jobs fulfilled in job " << meta->id << std::endl;
    }

    return false;
}

local_task_scheduler::local_task_scheduler(int id, meta_task_scheduler *global)
    : id(id), size(0), global(global), looking_for_task(false), meta_tasks_scheduled_here(0)
{
    pthread_mutex_init(&local_queue_mutex, 0);
}

local_task_scheduler::~local_task_scheduler()
{
    pthread_mutex_destroy(&local_queue_mutex);
}


bool local_task_scheduler::getTask(local_task *out)
{
    pthread_mutex_lock(&local_queue_mutex);
    if ((local_queue.size() < size) && !looking_for_task)  // size => LOWER_LOCAL_WATERMARK
    {
        looking_for_task = true;                  // is set while holding the mutex ->no race condition
        pthread_mutex_unlock(&local_queue_mutex); // there might still be jobs left in the queue

        fillLocalQueue();

        pthread_mutex_lock(&local_queue_mutex);   // (act as if this did not have to refill the queue)
    }

    // we still have the mutex
    if (local_queue.empty())
    {
        // somebody must be looking for a new meta_task so don't hog the mutex
        pthread_mutex_unlock(&local_queue_mutex);
        return false;
    }

    (*out) = local_queue.front();
    local_queue.pop_front();

    pthread_mutex_unlock(&local_queue_mutex);
    return true;
}

void local_task_scheduler::fillLocalQueue()
{
    meta_task_ptr meta;
    while (local_queue.size() < size)  // size => HIGHER_LOCAL_WATERMARK
        // we do not hold the mutex but it should be fine here
        // worst case we overestimate local work ?
    {
        if (!global->getMetaTask(id, &meta))
        {
            looking_for_task = false;
            return;
        }
        meta_tasks_scheduled_here++;
        segmentMetaTask(meta);
    }
    looking_for_task = false;
}


void local_task_scheduler::segmentMetaTask(meta_task_ptr meta)
{
    // there should be no race condition since meta is only accessed by one thread
    meta->fulfilled.store(0, std::memory_order_relaxed);

    pthread_mutex_lock(&local_queue_mutex);
    switch (meta->type)
    {
    case PANEL:
        {
        int i = 0;
        for (int k = 0; k < meta->meta_y && i < size; k+=4) local_queue.push_back(local_task(i++, meta));
        meta->nlocal = i;
        break;
        }
    case BEHIND_PANEL_UPDATE:
        for (int i = 0; i < meta->meta_x; i++)
        {
            local_queue.push_back(local_task(i,meta));
        }
        meta->nlocal = meta->meta_x;
        break;
    case B:
    case SCHUR_COMPLEMENT:
    case RANDOMIZE:
        for (int i = 0; i < meta->meta_x*meta->meta_y; i++)
        {
            local_queue.push_back(local_task(i,meta));
        }
        meta->nlocal = meta->meta_x*meta->meta_y;
        break;

    default:
        std::cout << "error: unknown task type " << meta->type;
    }    
    meta->status = LOCAL_QUEUE;
    pthread_mutex_unlock(&local_queue_mutex);
}
